<?php

namespace Controller;

use Firebase\JWT\JWT;
use Inc\Helper;
use Inc\Logger;

/**
 * Class Base
 * @package Controller
 */
abstract class Base {
  protected $view;
  private $user;
  private $f3;

  /**
   * Before route
   */
  public function beforeroute($f3, $params) {
    $this->f3 = $f3;
    
    // if setup is not complete reroute to installation
    if ($f3->get('INSTALL.STEP') > 0 && $f3->get('ALIAS') != 'install')
      $f3->reroute('@install');

    // init default view
    $this->view = new \View\Page();

    $this->setLanguage();
    $this->setUser();

    // make certain config variables an array if it contains only one value
    foreach (['SITE.COINS_FIELDS', 'SITE.RISERS_FALLERS_FIELDS', 'SITE.COIN_TABS', 'SITE.DONATION_ADDRESSES'] as $field) {
      if ($f3->get($field) && !is_array($f3->get($field)))
        $f3->set($field, [$f3->get($field)]);
    }
  }

  /**
   * After route
   */
  public function afterroute($f3, $params) {
    if (!$this->view)
      trigger_error('No View has been set.');

    $this->view->format();
    print $this->view->render();
  }

  public function user() {
    return $this->user;
  }
  
  protected function setUser() {
    // check if user is logged in
    $authCookie = isset($_COOKIE['cc_token']) ? $_COOKIE['cc_token'] : NULL;
    $securityToken = $this->f3->get('SECURITY.TOKEN');
    if ($authCookie && $securityToken) {
      try {
        $jwt = JWT::decode($_COOKIE['cc_token'], $this->f3->get('SECURITY.TOKEN'), [$this->f3->get('SECURITY.ALGORITHM')]);
        $this->user = (array) $jwt->user;
        $this->f3->set('USER', $this->user);
      } catch (\Firebase\JWT\SignatureInvalidException $e) {
        setcookie('cc_token', '', 0);
        Logger::log(sprintf('Failed to decode JWT: %s', $_COOKIE['cc_token']), 2);
      } catch (\Firebase\JWT\ExpiredException $e) {
        setcookie('cc_token', '', 0);
        Logger::log(sprintf('JWT expired: %s', $_COOKIE['cc_token']), 2);
      } catch (\Firebase\JWT\BeforeValidException $e) {
        Logger::log(sprintf('JWT is not yet valid: %s', $_COOKIE['cc_token']), 2);
      }
    }   
  }
  
  protected function setLanguage() {
    // check if user set custom language in the front-end
    if (isset($_COOKIE['language'])) {
      $this->f3->set('LANGUAGE', $_COOKIE['language']);
      $this->f3->set('LANG', $_COOKIE['language']); // set custom LANG variable, because LANGUAGE returns a list, e.g.: en-US,en
      $this->f3->set('LANGSHORT', substr($_COOKIE['language'], 0, 2));
      // otherwise you the system default
    } else {
      $this->f3->set('LANG', explode(',', $this->f3->get('LANGUAGE'))[0]); // set custom LANG variable, because LANGUAGE returns a list, e.g.: en-US,en
      $this->f3->set('LANGSHORT', explode(',', $this->f3->get('LANGUAGE'))[1]);
    }

    // it's important to use fixed locale for numbers,
    // otherwise such functions like json_encode, json_decode can change number format implicitly
    setlocale(LC_NUMERIC, 'C');
  }
}