<?php

namespace Controller;

use Inc\Config;
use Inc\Database;
use Inc\Helper;
use Inc\Flash;
use Inc\Logger;
use Inc\Phinx;
use Model\CurrencyRate;
use Model\Query;
use Model\User;
use Model\Action;

class Installer extends Base {
  private $step;

  /**
   * Before a page is served
   * @param $f3
   * @param $params
   */
  public function beforeroute($f3, $params) {
    parent::beforeroute($f3, $params);
    $this->view->templateFolder = 'install';

    $this->step = $f3->get('INSTALL.STEP');
    if ($this->step == 0)
      $f3->reroute('@index');
  }

  /**
   * Install page
   * @param $f3
   * @param $params
   */
  public function install($f3, $params) {

    // User submits form by pressing Next button
    if ($f3->get('VERB')=='POST') {
      // Create database tables and insert data
      if ($this->step == 1) {
        $code = $f3->get('POST.code');
        $email = $f3->get('POST.email');

        $this->view->data['email'] = $email;

        Config::instance()->update([
            'ENVATO#PURCHASE_CODE' => $code,
        ]);

        try {
          $response = \Web::instance()->request('https://financialplugins.com/api/licenses/register', [
              'method'  => 'POST',
              'content' => [
                'code' => $code,
                'email' => $email,
                'domain' => trim(Helper::baseUrl(), '/'),
                'hash' => $f3->get('HASH')
              ]
          ]);
        } catch (\Exception $e) {
          Logger::log(sprintf('%s: %s', get_class($e), $e->getMessage()));
          Flash::instance()->addMessage('There was an error while registering your license.', 'error');
        }

        $result = json_decode($response['body']);
        if (!$result->success) {
          Flash::instance()->addMessage($result->message, 'error');
        } else {
          Config::instance()->update([
              'ENVATO#SECURITY_HASH' => $result->message,
              'INSTALL#STEP' => ++$this->step
          ]);
        }
        // Create database tables and insert data
      } elseif ($this->step == 2) {
        $this->view->data['host'] = $f3->get('POST.host');
        $this->view->data['port'] = $f3->get('POST.port');
        $this->view->data['database'] = $f3->get('POST.database');
        $this->view->data['user'] = $f3->get('POST.user');
        $this->view->data['password'] = $f3->get('POST.password');

        if (Config::instance()->update([
          'DB#HOST'     => $f3->get('POST.host'),
          'DB#PORT'     => $f3->get('POST.port'),
          'DB#DATABASE' => $f3->get('POST.database'),
          'DB#USER'     => $f3->get('POST.user'),
          'DB#PASSWORD' => $f3->get('POST.password')
        ])) {
          try {
            // if parameters are not correct PDOException will be thrown
            $db = Database::instance();
            $phinx = new Phinx();

            $phinx->rollback();
            $phinx->migrate();

            $query = new Query();
            if ($query->countCoins() == 0) {
              $phinx->seedCoin();
            }

            if ($query->countCurrencies() == 0) {
              $phinx->seedCurrency(TRUE);
            }

            // increment step and save it
            Config::instance()->update(['INSTALL#STEP' => ++$this->step]);
          } catch (\PDOException $e) {
            Logger::log(sprintf('%s: %s', get_class($e), $e->getMessage()));
            Flash::instance()->addMessage('There was an error while connecting to the database. Please check the parameters and try again.', 'error');
          } catch (\Exception $e) {
            Logger::log(sprintf('%s: %s', get_class($e), $e->getMessage()));
            Flash::instance()->addMessage('There was an error while setting up MySQL tables.', 'error');
          }
        }
      // Create admin user
      } elseif ($this->step == 3) {
        $this->view->data['email'] = $f3->get('POST.email');
        $this->view->data['user_password'] = $f3->get('POST.user_password');

        $user = new User();
        if ($user->create([
            'first_name'  => 'Admin',
            'last_name'   => 'Admin',
            'email'       => $this->view->data['email'],
            'password'    => $this->view->data['user_password'],
            'admin'       => 1
          ])->lastId()) {
          // increment step and save it
          Config::instance()->update(['INSTALL#STEP' => ++$this->step]);
        } else {
          Flash::instance()->addMessage('There was an error while setting up the admin user.', 'error');
        }
      // Retrieve and save current market data
      } elseif ($this->step == 4) {
        set_time_limit(1800); // 30 min

        $currency = $f3->get('POST.currency');
        $cryptocompareKey = $f3->get('POST.cryptocompare_key');
        $openexchangeratesKey = $f3->get('POST.openexchangerates_key');
        Config::instance()->update([
          'SITE#CURRENCY' => $currency,
          'API#CRYPTOCOMPARE_API_KEY' => $cryptocompareKey,
          'API#OPENEXCHANGERATES_APP_ID' => $openexchangeratesKey
        ]);
        $f3->set('SITE.CURRENCY', $currency);

        $action = new Action();
        // update currencies data
        $this->view->data['updated_currencies_count'] = $action->updateCurrencyRates();
        // update coins data
        $this->view->data['updated_coins_count'] = $action->updateCoinData();

        Logger::log(sprintf('Data for %d currencies updated', $this->view->data['updated_currencies_count']));
        Logger::log(sprintf('Data for %d coins updated', $this->view->data['updated_coins_count']));
        if ($this->view->data['updated_coins_count']) {
          // increment step and save it
          $this->step = 0;
          Config::instance()->update([
            'SECURITY#TOKEN' => Helper::secureToken(),
            'INSTALL#STEP' => $this->step
          ]);
        } else {
          Flash::instance()->addMessage('There was an error while pulling and saving market data.', 'error');
        }
      }
    }

    // check system requirements in the first step
    if ($this->step == 1) {
      $this->view->data['env_issues'] = $this->checkEnvironment();
    } elseif ($this->step == 4) {
      $currencyRate = CurrencyRate::instance();
      $this->view->data['currencies_list'] = $currencyRate->all();
    }

    $this->view->data['step'] = $this->step;
  }

  /**
   * Various checks to ensure the installer can proceed
   * @return array - list of env issues
   */
  private function checkEnvironment() {
    $issues = [];

    foreach (['app/data/history', 'app/dict-udf', 'assets/images/coins', 'db/seeds', 'config', 'logs', 'sitemap', 'tmp'] as $folder) {
      if (!is_writeable($folder))
        $issues[] = sprintf('The following folder should be writeable by the web server (www-data user): %s', $folder);
    }

    if ($disabled_functions = ini_get('disable_functions')) {
      foreach (explode(',', $disabled_functions) as $function) {
        if (in_array($function, ['set_time_limit', 'php_uname']))
          $issues[] = sprintf('PHP function "%s" should be allowed. Please remove it from php.ini disable_functions variable. You might need to contact your hosting support to do it.', $function);
      }
    }

    return $issues;
  }
}