<?php

namespace Inc;

use Model\Query;

class Helper {
  public static $languagesNativeNames = array(
    'en-US' => 'English',
    'de-DE' => 'Deutsch',
    'es-ES' => 'Español',
    'fr-FR' => 'Français',
    'pt-BR' => 'Português',
    'pt-PT' => 'Português',
    'nl-NL' => 'Nederlands',
    'ru-RU' => 'Русский',
    'cs-CZ' => 'Česky',
    'it-IT' => 'Italiano',
    'fi-FI' => 'Suomi',
    'sv-SE' => 'Svenska',
    'hu-HU' => 'Magyar',
    'el-GR' => 'Ελληνικά',
    'da-DK' => 'Dansk',
    'lv-LV' => 'Latviešu',
    'lt-LT' => 'Lietuvių',
    'et-EE' => 'Eesti',
    'sk-SK' => 'Slovenčina',
    'sl-SI' => 'Slovenščina',
  );

  /**
   * Return host name and path to index.php, for example http://site.com/webroot/ or http://site.com/ when installed to root folder
   * @return string
   */
  public static function baseUrl() {
    return sprintf(
      "%s://%s%s",
      self::protocol(),
      $_SERVER['SERVER_NAME'],
      $_SERVER['PHP_SELF']=='/index.php' ? '/' : dirname($_SERVER['PHP_SELF']).'/'
    );
  }

  /**
   * Get current page URI (relative to base path)
   * @return mixed
   */
  public static function uri() {
    $f3 = \Base::instance();
    return $f3->get('BASE') ? str_replace($f3->get('BASE'), '', $f3->get('URI')) : $f3->get('URI');
  }

  public static function protocol() {
    return (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off') ||
      (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && strtolower($_SERVER['HTTP_X_FORWARDED_PROTO'])=='https') ?
        'https' :
        'http';
  }
  /**
   * Return current timestamp in a given timezone or default website timezone and optionally convert to another timezone
   * @param string $time
   * @param int $offset
   * @param null $inputTimezone
   * @return string
   */
  public static function timeStamp($time = 'now', $offset = NULL, $inputTimezone = NULL, $outputTimezone = NULL, $format = 'Y-m-d H:i:s') {
    $offset = intval($offset);
    $datetime = $inputTimezone ? new \DateTime($time, new \DateTimeZone($inputTimezone)) : new \DateTime($time);
    // apply offset in seconds
    if ($offset>0) {
      $datetime->add(new \DateInterval('PT'.$offset.'S'));
    } elseif ($offset<0) {
      $datetime->sub(new \DateInterval('PT'.abs($offset).'S'));
    }
    if ($outputTimezone) {
      $outTz = new \DateTimeZone($outputTimezone);
      $datetime->setTimezone($outTz);
    }
    return $datetime->format($format);
  }

  /**
   * Generate random token
   * @return string
   */
  public static function token() {
    return bin2hex(openssl_random_pseudo_bytes(16));
  }

  public static function p($msg) {
    if (is_array($msg) || is_object($msg)) {
      print '<pre>'.print_r($msg, TRUE).'</pre>';
    } else {
      print $msg.'<br>';
    }
  }

  /**
   * Format number
   * @param $input
   * @param int $decimals
   * @return string
   */
  public static function formatNumber($input, $decimals = 0, $prefix = '', $suffix = '') {
    $input = floatval($input);
    $absInput = abs($input);
    if ($decimals === 'auto') {
      if ($absInput >= 0.01) {
        $decimals = 2;
      } elseif (0.0001 <= $absInput && $absInput < 0.01) {
        $decimals = 4;
      } elseif (0.000001 <= $absInput && $absInput < 0.0001) {
        $decimals = 6;
      } elseif ($absInput < 0.000001) {
        $decimals = 8;
      }
    }
    return ($prefix ? $prefix : '') . number_format($input, $decimals, self::decimalPoint(), self::thousandsSeparator()) . ($suffix ? $suffix : '');
  }

  public static function thousandsSeparator() {
    $f3 = \Base::instance();
    return $f3->get('SITE.THOUSANDS_SEPARATOR') ? chr($f3->get('SITE.THOUSANDS_SEPARATOR')) : ',';
  }

  public static function decimalPoint() {
    $f3 = \Base::instance();
    return $f3->get('SITE.DECIMAL_POINT') ? chr($f3->get('SITE.DECIMAL_POINT')) : '.';
  }

  /**
   * Convert "first_second" to "FirstSecond"
   * @param $string
   * @return mixed
   */
  public static function snakeToCamelCase($string) {
    return str_replace(' ','',ucwords(str_replace('_',' ',$string)));
  }

  /**
   * Transform slug as page title, e.g. about-us --> About Us
   * @param $slug
   */
  public static function slugToTitle($slug) {
    return ucwords(str_replace('-',' ',$slug));
  }

  /**
   * Generate secure random token (taken from Laravel framework)
   * @param int $length
   * @return string
   */
  public static function secureToken($length = 32) {
    if (function_exists('openssl_random_pseudo_bytes')) {
      if ($bytes = openssl_random_pseudo_bytes($length * 2)) {
        return substr(str_replace(['/', '+', '='], '', base64_encode($bytes)), 0, $length);
      }
    }

    return self::quickRandom($length);
  }

  public static function quickRandom($length = 32) {
    $pool = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    return substr(str_shuffle(str_repeat($pool, 5)), 0, $length);
  }


  /**
   * Check if GD is enabled
   * @return bool
   */
  public static function gdEnabled() {
    return extension_loaded('gd');
  }

  /**
   * Check if imagick is installed and enabled
   * @return bool
   */
  public static function imagickEnabled() {
    return extension_loaded('imagick') && class_exists('Imagick');
  }

  /**
   * Check if fileinfo module is enabled
   * @return bool
   */
  public static function finfoEnabled() {
    return function_exists('finfo_buffer');
  }

  /**
   * Get coin symbol based on the coin page key in the URL (e.g. /coin/bitcoin)
   * @param $key
   * @return string
   */
  public static function coinSymbol($key) {
    $f3 = \Base::instance();

    $coinPageKey = $f3->get('SITE.COIN_PAGE_KEY');

    switch($coinPageKey) {
      case 'lcn':
        $coins = (new Query())->coinsList();
        foreach($coins as $coin)
          if(preg_replace('/[^a-z0-9]{1,}/', '-', strtolower($coin['name'])) == $key)
            return $coin['symbol'];
        break;
      case 'lcs':
        return strtoupper($key);
        break;
      case 'ucs':
        return $key;
    }
  }

  public static function coinPageKey($coin) {
    $f3 = \Base::instance();

    $coinPageKey = $f3->get('SITE.COIN_PAGE_KEY');

    switch($coinPageKey) {
      case 'lcn': return preg_replace('/[^a-z0-9]{1,}/','-',strtolower($coin['name']));
      case 'lcs': return strtolower($coin['symbol']);
      case 'ucs': return $coin['symbol'];
    }
  }

  /**
   * Get all supported languages, e.g. en-US, fr-FR etc
   * @return array
   */
  public static function languages() {
    $result = [];
    $f3 = \Base::instance();
    // take into account folder with user defined language files
    $dictFolders = explode(';', $f3->get('LOCALES'));
    foreach (glob($dictFolders[1].'*.php') as $file) {
      if (preg_match('#([a-z]{2}\-[A-Z]{2})\.php$#', $file, $matches)){
        $result[$matches[1]] = isset(self::$languagesNativeNames[$matches[1]]) ? self::$languagesNativeNames[$matches[1]] : $matches[1];
      }
    }
    return array_unique($result);
  }

  /**
   * Get sign of a number
   * @param $n
   * @return bool
   */
  public static function sign($n) {
    return ($n > 0) - ($n < 0);
  }
}