<?php

namespace Model;
use Inc\HTTP;
use Inc\Image;
use Inc\Logger;
use Inc\Helper;

/**
 * Class Action - generic model to insert/update data, perform calclulations etc
 * @package Model
 */
class Action extends Base {
	const CRYPTOCOMPARE_URL = 'https://www.cryptocompare.com';
	const COINS_STATIC_DATA_URL = 'https://min-api.cryptocompare.com/data/all/coinlist?api_key=%s';
	const COINS_INFO_URL = 'https://www.cryptocompare.com/api/data/coinsnapshotfullbyid/?id=%d';
	const COINS_STATIC_DATA_FILE = 'db/seeds/coins.json';
	const COINS_IMAGES_DIR = 'assets/images/coins';
	const COINS_THUMBS_DIR = 'assets/images/coins/thumb30';
	const COINS_MARKET_DATA_URL = 'https://min-api.cryptocompare.com/data/pricemultifull?fsyms=%s&tsyms=%s&api_key=%s';
	const CURRENCIES_MARKET_DATA_URL = 'https://openexchangerates.org/api/latest.json?app_id=%s';
	const CURRENCIES_MARKET_DATA_CACHE_TIME = 3600; // 1 hour
	
	
	public function __construct() {
		parent::__construct();
	}

	/**
	 * Get FX rates from API and update the local database
	 * @return int|mixed - updated records count
	 */
	public function updateCurrencyRates() {
		$updatedCurrenciesCount = 0;
		$apiKey = $this->f3->get('API.OPENEXCHANGERATES_APP_ID');
		if ($apiKey && (time() - CurrencyRate::instance()->lastUpdated() >= self::CURRENCIES_MARKET_DATA_CACHE_TIME)) {
			$requestUrl = sprintf(self::CURRENCIES_MARKET_DATA_URL, $apiKey);
			Logger::log($requestUrl, 3);
			if ($response = HTTP::get($requestUrl)) {
				$json = json_decode($response);
				if (isset($json->rates)) {
					foreach ($json->rates as $symbol => $rate) {
						$updatedCurrenciesCount += (new Currency($symbol))->update(['rate' => $rate]);
					}
				} else {
					Logger::log('Unexpected response from openexchangerates API');
					Logger::log($response);
				}
			}
		}
		return $updatedCurrenciesCount;
	}

	/**
	 * Get cryptocurrency market data from API and update the local database
	 * @return int|mixed - updated records count
	 */
	public function updateCoinData() {
		$updatedCoinsCount = 0;
		$symbols = explode(',', (new Query())->symbols());
		Logger::log(sprintf('Start updating %d coins', count($symbols)));
		$currency = 'USD'; // always store coins data in USD and convert to other currencies on demand
		foreach (array_chunk($symbols, 50) as $i => $symbols_chunk) {
			$requestUrl = sprintf(self::COINS_MARKET_DATA_URL, implode(',', array_map('urlencode', $symbols_chunk)), $currency, $this->f3->get('API.CRYPTOCOMPARE_API_KEY'));
			Logger::log(sprintf('%d. %s', ($i+1), $requestUrl), 3);
			if ($response = HTTP::get($requestUrl)) {
				if ($data = json_decode($response)) {
					if (!isset($data->Response) || $data->Response != 'Error') {
						foreach ($data->RAW as $symbol => $coin) {
							$data = [
								'price' => $coin->$currency->PRICE,
								'change' => $coin->$currency->CHANGE24HOUR,
								'change_pct' => $coin->$currency->CHANGEPCT24HOUR,
								'open' => $coin->$currency->OPEN24HOUR,
								'low' => $coin->$currency->LOW24HOUR,
								'high' => $coin->$currency->HIGH24HOUR,
								'supply' => $coin->$currency->SUPPLY,
								'market_cap' => $coin->$currency->MKTCAP,
								'volume' => $coin->$currency->VOLUME24HOUR,
								'volume_ccy' => $coin->$currency->VOLUME24HOURTO,
								'total_volume' => $coin->$currency->TOTALVOLUME24H,
								'total_volume_ccy' => $coin->$currency->TOTALVOLUME24HTO,
								'last_updated' => Helper::timeStamp()
							];
							Logger::log(array_merge($data, ['symbol' => $symbol]), 3);
							$updatedCoinsCount += (new Coin($symbol))->update($data);
						}
					} else {
						Logger::log(sprintf('Web request error: %s', $data->Message));
					}
				}
			} else {
				Logger::log('Empty response!');
			}
		}
		return $updatedCoinsCount;
	}

	/**
	 * Download new coins, including logo and extra information and save to a local JSON file
	 *
	 * @param $updateSubs - update coins subscriptions for existing coins
	 * @return int
	 */
	public function downloadCoins($updateSubs = FALSE) {
		$newCoinsCount = 0;

		if ($coinDataResponse = HTTP::get(sprintf(self::COINS_STATIC_DATA_URL, $this->f3->get('API.CRYPTOCOMPARE_API_KEY')))) {
			if ($coinDataResponse = json_decode($coinDataResponse)) {
				if (isset($coinDataResponse->Response) && $coinDataResponse->Response == 'Success') {
					$coins = json_decode(file_get_contents(self::COINS_STATIC_DATA_FILE), JSON_OBJECT_AS_ARRAY);
					// loop through each retrieved coin
					foreach ($coinDataResponse->Data as $coinSymbol => $coin) {
						$coinId = $coin->Id;
						$coinSymbol = trim($coinSymbol);
						$coinName = trim($coin->CoinName);
						$coinAttributesUpdated = FALSE;
						$newCoin = FALSE;

						// if such coin exists in local JSON files
						if (isset($coins[$coinId])) {
							// update symbol
							if ($coins[$coinId]['symbol'] != $coinSymbol) {
								$coins[$coinId]['symbol'] = $coinSymbol;
								$coinAttributesUpdated = TRUE;
							}
							// update name
							if ($coins[$coinId]['name'] != $coinName) {
								$coins[$coinId]['name'] = $coinName;
								$coinAttributesUpdated = TRUE;
							}
						// new coin
						} else {
							Logger::log(sprintf('%s: new coin "%s" found (id %d)', $coinSymbol, $coinName, $coinId), 2);
							Logger::log($coin, 2);

							$newCoin = TRUE;

							$coins[$coinId] = [
								'id'      => $coinId,
								'symbol'  => $coinSymbol,
								'name'    => $coinName,
							];

							$replacements = [
								'<p> </p>',
								'<p></p>',
								'<strong> </strong>',
								'<strong></strong>'
							];

							$newCoinsCount++;
						}

						// download detailed coin info for new coins or if subs updated is requested
						if ($newCoin || $updateSubs) {
							Logger::log(sprintf('%s: retrieving detailed info (id %d)', $coinSymbol, $coinId), 2);
							if ($coinInfoResponse = HTTP::get(sprintf(self::COINS_INFO_URL, $coinId))) {
								if ($coinInfoResponse = json_decode($coinInfoResponse)) {
									if (isset($coinInfoResponse->Response) && $coinInfoResponse->Response == 'Success') {
										// get description, website etc ONLY for new coins
										if ($newCoin) {
											$info = $coinInfoResponse->Data->General;
											Logger::log($coinInfoResponse->Data, 2);
											if ($info->ProofType) {
												$coins[$coinId]['proof_type'] = $info->ProofType;
											}
											if ($info->AffiliateUrl != '-') {
												$coins[$coinId]['website'] = str_replace('-', '', $info->AffiliateUrl);
											}
											if ($info->Twitter) {
												$coins[$coinId]['twitter'] = str_replace('@', '', $info->Twitter);
											}
											if ($info->Description) {
												$coins[$coinId]['description'] = str_replace($replacements, '', $info->Description);
											}
											if ($info->Features) {
												$coins[$coinId]['features'] = str_replace($replacements, '', $info->Features);
											}
											if ($info->Technology) {
												$coins[$coinId]['technology'] = str_replace($replacements, '', $info->Technology);
											}
											if ($info->TotalCoinSupply) {
												$coins[$coinId]['total_supply'] = intval($info->TotalCoinSupply);
											}
										}

										// update subs for new coins or if subs updated is requested
										if ($newCoin || $updateSubs) {
											$subs = $coinInfoResponse->Data->Subs;
											if (!empty($subs)) {
												array_walk($subs, function(&$sub) {
													// $sub will look like this "2~Bleutrade~PPC~BTC"
													// remove sub type from the beginning of the string
													$sub = substr($sub, strpos($sub, '~') + 1);
												});
												$coins[$coinId]['subs'] = implode(',', $subs);
											}
										}
									}
								}
							}
						}

						// save coin logo (check even for existing coins)
						if (isset($coin->ImageUrl)) {
							$logoExtension = substr($coin->ImageUrl, strrpos($coin->ImageUrl, '.'));
							$logoFileName = $coinId . '-' . preg_replace('#[^a-z0-9]#i', '', $coinSymbol) . $logoExtension;
							$logoFilePath = self::COINS_IMAGES_DIR . '/' . $logoFileName;
							// download logo if it doesn't exist or coin symbol/name was changed
							if (!file_exists($logoFilePath) || $coinAttributesUpdated) {
								Logger::log(sprintf('%s: saving logo from %s', $coinSymbol, $coin->ImageUrl), 2);
								if (($imageResponse = HTTP::get(self::CRYPTOCOMPARE_URL . $coin->ImageUrl)) && strpos($imageResponse, '!DOCTYPE html')===FALSE) {
									$image = new Image($imageResponse);
									if ($image->save($logoFilePath)) {
										$coins[$coinId]['logo'] = $logoFileName;
										// create logo thumb (height = 30px)
										$image->thumb(self::COINS_THUMBS_DIR . '/' . $logoFileName, NULL, 30);
									}
								}
							} else {
								Logger::log(sprintf('%s: Logo already exists', $coinSymbol), 2);
								$coins[$coinId]['logo'] = $logoFileName;
							}
						}
					}

					// check if some coins were removed from the API, in this case delete them from coins.json file
					foreach ($coins as $coinId => $coin) {
						$symbol = $coin['symbol'];
						if (!isset($coinDataResponse->Data->$symbol) || (isset($coinDataResponse->Data->$symbol) && $coinId != $coinDataResponse->Data->$symbol->Id))
							unset($coins[$coinId]);
					}

					// save coins to a local file
					file_put_contents(self::COINS_STATIC_DATA_FILE, json_encode($coins, JSON_PRETTY_PRINT));
					Logger::log(sprintf('New coins added: %d', $newCoinsCount));
				}
			}
		}
		return $newCoinsCount;
	}
	
	public function updateSitemap(){
		$baseUrl = substr(Helper::baseUrl(), 0, strlen(Helper::baseUrl())-1);

		if(!file_exists('sitemap/sitemap.xml')){
			$xml='<?xml version="1.0" encoding="UTF-8"?><sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"><sitemap><loc>'.$baseUrl.'/sitemap/sitemap.coins.xml</loc></sitemap><sitemap><loc>'.$baseUrl.'/sitemap/sitemap.system.xml</loc></sitemap><sitemap><loc>'.$baseUrl.'/sitemap/sitemap.udf.xml</loc></sitemap></sitemapindex>';
			file_put_contents('sitemap/sitemap.xml',$xml);
		}
		
		if(!file_exists('sitemap/sitemap.coins.xml')|| filectime('sitemap/sitemap.coins.xml') < time()-24*60*60){
			global $f3;
			$d = date('c');
			$xml = '<?xml version="1.0" encoding="UTF-8"?><urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
			$query = new Query();
	    	$coins = $query->coinsListFull('active=1');
			foreach($coins as $coin){
				$link = $baseUrl . $f3->alias('coin','symbol=' . Helper::coinPageKey($coin));
				$priority = 0.5;
				if ($coin['featured']) $priority=0.7;
				$xml.="<url><loc>$link</loc><lastmod>$d</lastmod><changefreq>daily</changefreq><priority>$priority</priority></url>";
				
			}
			
			//coin_page_key
			
			$xml.='</urlset>';
			
			file_put_contents('sitemap/sitemap.coins.xml',$xml);
		}
		
		if(!file_exists('sitemap/sitemap.system.xml')){
			global $f3;
			$xml = '<?xml version="1.0" encoding="UTF-8"?><urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
			
			$links=[
				'index',
				'risers',
				'fallers',
				'favorites',
				'calculator',
				'widget',
				'login',
				'register',
				'password'
			];
			
			$priority=0.8;
			
			foreach($links as $link){
				$link = $baseUrl . $f3->alias($link);
				$xml .= "<url><loc>$link</loc><changefreq>daily</changefreq><priority>$priority</priority></url>";
			}
			
			$xml.='</urlset>';
			file_put_contents('sitemap/sitemap.system.xml',$xml);
		}

		if(!file_exists('sitemap/sitemap.udf.xml')){
			$xml = '<?xml version="1.0" encoding="UTF-8"?><urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
			$xml.='</urlset>';
			file_put_contents('sitemap/sitemap.udf.xml',$xml);
		}
	}
}