<?php

namespace Model;
use Inc\Logger;
use Inc\Helper;

/**
 * Class CurrencyRate - Singleton to retrieve and provide current currency rates
 * It should be called like this: CurrencyRate::instance()->get()
 * @package Helper
 */
class CurrencyRate extends BaseSingleton {
  private $currencies;
  private $symbol;

  // using protected constructor for singleton, so it can't be instantiated like a normal class
  protected function __construct() {
    parent::__construct();
    $this->symbol = isset($_COOKIE['currency']) ? $_COOKIE['currency'] : $this->f3->get('SITE.CURRENCY');
    foreach ($this->db->exec('SELECT symbol, code, name, rate, last_updated FROM currency') as $row) {
      $this->currencies[$row['symbol']] = $row;
    }

    return $this;
  }

  /**
   * Get all currencies and their rates
   * @return mixed
   */
  public function all() {
    return $this->currencies;
  }

  /**
   * Get current user currency array
   * @return null
   */
  public function get() {
    return isset($this->currencies[$this->symbol]) ? $this->currencies[$this->symbol] : NULL;
  }

  /**
   * Get FX rate of a user currency (against the base website currency - SITE.CURRENCY)
   * @return number
   */
  public function rate() {
    return isset($this->currencies[$this->symbol]['rate']) ? $this->currencies[$this->symbol]['rate'] : 0;
  }

  /**
   * Get current user currency symbol
   * @return string, e.g. USD, EUR etc
   */
  public function symbol() {
    return $this->symbol;
  }

  /**
   * Get last updated date & time
   * @return mixed
   */
  public function lastUpdated() {
    return max(array_map('strtotime', array_column($this->currencies, 'last_updated')));
  }

  /**
   * Get current user currency code
   * @return string, e.g. $, AU$ etc
   */
  public function code() {
    return isset($this->currencies[$this->symbol]['code']) ? $this->currencies[$this->symbol]['code'] : '';
  }
}