<?php

namespace Model;

use DB\SQL\Mapper;
use Inc\Helper;
use Inc\Logger;

class User extends Base {
  private $mapper;

  public function __construct() {
    parent::__construct();
    $this->mapper = new Mapper($this->db->connection(), 'user');
    return $this;
  }

  /**
   * Log user in
   *
   * @param $email
   * @param $password
   * @return \Model\User
   */
  public function login($email, $password) {
    Logger::log(sprintf('Login attempt for user %s', $email), 2);

    $user = $this->load('email', $email)->get();
    if (!empty($user)) {
      if (password_verify($password, $user['password'])) {
        Logger::log(sprintf('User %d logged in', $user['id']), 2);
        $this->update(['last_login' => Helper::timeStamp()]);
      } else {
        Logger::log(sprintf('Incorrect password provided by %s', $email), 2);
        $this->reset();
      }
    }

    return $this;
  }

  /**
   * Check if a user account with certain email exists
   * @param $mail
   * @return bool
   */
  public function count($key, $value) {
    return $this->mapper->count([$key.' = :'.$key, ':'.$key => $value]);
  }

  /**
   * Load user by given key (id, email) and value
   * @param $key
   * @param $value
   * @return mixed
   */
  public function load($key, $value) {
    $this->mapper->load([$key.' = :'.$key, ':'.$key => $value]);
    return $this;
  }

  /**
   * Get loaded user
   * @return array|null
   */
  public function get() {
    return !$this->mapper->dry() ? $this->mapper->cast() : NULL;
  }

  public function reset() {
    $this->mapper->reset();
  }

  /**
   * Update existing or create a new record
   *
   * @param $fields
   * @return $this
   */
  public function update($fields) {
    $this->mapper->copyfrom($fields);
    $this->mapper->save();
    return $this;
  }

  /**
   * Get the ID of the last inserted row
   * @return mixed
   */
  public function lastId() {
    return $this->mapper->get('_id');
  }

  /**
   * Create a new user
   *
   * @param $params
   * @return \Model\User
   */
  public function create($params) {
    Logger::log(sprintf('Creating user: %s, %s, %s', $params['first_name'], $params['last_name'], $params['email']), 2);

    $fields = [
      'first_name' => $params['first_name'],
      'last_name'  => $params['last_name'],
      'email'      => $params['email'],
      'password'   => password_hash($params['password'], PASSWORD_BCRYPT),
      'admin'      => $params['admin'],
      'created'    => Helper::timeStamp()
    ];
    if ($userId = $this->update($fields)->lastId()) {
      Logger::log(sprintf('User created. Email: %s, user ID: %d.', $params['email'], $userId), 2);
    }

    return $this;
  }
}